/**
* \file: tst_bmpdec_linux.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: svg_bmpdec
*
* \copyright (c) 2010-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "EGL/egl.h"
#include "GLES2/gl2.h"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <adit_typedef.h>
#include <svg_bitmap_decoder.h>


#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480


#define PNG_MAX_SIZE_SMALL  (2048*2048*3)
#define NUM_POS             (5)
#define SMALL_CLIP          (32)
#define HALF_SMALL_CLIP     (SMALL_CLIP/2)


static SVGChar string[100];

static EGLDisplay       eglDisplay=EGL_NO_DISPLAY;
static EGLConfig        eglConfig;
static EGLContext       eglContext=EGL_NO_CONTEXT;
static EGLSurface       eglWindowSurface=EGL_NO_SURFACE;

static SVGContextBmpDec *gp_bmp_ctx = NULL;

/* Local declarations */
static SVGInt32 tst_bmpdec_read_img(const SVGChar* p_filename,SVGUint8* p_imagedata);
static void tst_bmpdec_draw_image(SVGInt16* pos_x, SVGInt16* pos_y, SVGImage* svg_image);
static SVGInt32 tst_bmpdec_png_queued(void);
#ifdef COMMENTED_OUT
static SVGInt32 tst_bmpdec_png_invalid(void);
#endif
static SVGInt32 tst_bmpdec_formats(void);
static SVGInt32 tst_bmpdec_errors(void);
static SVGInt32 tst_bmpdec_mem_dest(void);
static SVGInt32 tst_bmpdec_async(void);

#ifdef USE_WAYLAND
	#include <wayland-client.h>
	#include <wayland-egl.h>
        #include <compositor-shim.h>
	#include "wayland-util.h"

typedef struct _WaylandGles WaylandGles;

struct _WaylandGles
{
  SVGUint32 surface_id;
  SVGUint32 layer_id;
  struct wl_display* wl_display;
  struct wl_surface* wl_surface;
  struct wl_egl_window* wl_native_window;
  struct wl_compositor* wl_compositor;
  struct wl_registry* wl_registry;

  struct compositor_shim_context* wlAdapterContext;
  struct compositor_shim_surface_context wlAdapterSurfaceContext;

};

static WaylandGles aWayland;

void RegistryHandleGlobal(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
  version = version;
  WaylandGles * sink = (WaylandGles *) data;

	if (!strcmp(interface, "wl_compositor")){
	    sink->wl_compositor =   (struct wl_compositor*)(wl_registry_bind(registry,
										  name,
										  &wl_compositor_interface,
										  1));
	}
}

static struct wl_registry_listener registryListener = {
    RegistryHandleGlobal,
    NULL
};


static SVGBoolean create_wl_context(WaylandGles * sink)
{
    sink->wl_display = wl_display_connect(NULL);
    if (sink->wl_display == NULL)
    {
  	    printf("fail wl 1\n");
        return FALSE;
    }

    sink->wl_registry = wl_display_get_registry(sink->wl_display);

    wl_registry_add_listener(sink->wl_registry, &registryListener, sink);
    wl_display_dispatch(sink->wl_display);
    wl_display_roundtrip(sink->wl_display);

    if(sink->wl_compositor == NULL)
    {
  	    printf("fail wl 2\n");
        return FALSE;
    }

    sink->wlAdapterContext = compositor_shim_initialize (sink->wl_display);
    if (NULL == sink->wlAdapterContext)
    {
        printf("fail wl 3\n");
        return FALSE;
    }
    sink->wl_surface = wl_compositor_create_surface(sink->wl_compositor);
    if(sink->wl_surface == NULL)
    {
        printf("fail wl 4\n");
        return FALSE;
    }

    sink->wl_native_window = wl_egl_window_create(sink->wl_surface, SCREEN_WIDTH, SCREEN_HEIGHT);
    if (sink->wl_native_window == NULL)
    {
        printf("fail wl 5\n");
        return FALSE;
    }

    return TRUE;
}

static void framelistener_cb(void *data, struct wl_callback *callback, uint32_t time_fl)
{
    data = data;
    time_fl = time_fl;

    if (callback)
    {
        wl_callback_destroy(callback);
    }
}

static const struct wl_callback_listener framelistener = {
    framelistener_cb
};

#else /* USE_WAYLAND */
#include <EGL/eglvivante.h>
#endif 

GLfloat projection[4][4] = {{2.0/800.0,0,0,0},{0,2.0/480.0,0,0},{0,0,0,0},{-1.0,-1.0,0,1}};

static char const * const apszFiles[2]=
{   /* shader for drawing glyps */
    "varying vec2 texcoord;                                       \
     attribute vec4 position;                                     \
     attribute vec2 inputtexcoord;                                \
     uniform mat4 mvp;                                            \
     void main(void)                                              \
     {                                                            \
        texcoord = inputtexcoord;                                 \
        gl_Position = mvp * position;                             \
     }",

    "varying highp vec2 texcoord;                                 \
     uniform sampler2D basetexture;                               \
     void main(void)                                              \
     {                                                            \
        gl_FragColor = texture2D(basetexture, texcoord);\
     }"
};

SVGInt32 hProgramHandle;
SVGInt32 mvplocation;
SVGInt32 texturelocation;
SVGInt32 vertexlocation;
SVGFloat modelview[4][4];
SVGFloat mvp[4][4];
SVGUint32 max_height = 0;

void local_matrixIdentity(SVGFloat pMatrix[4][4])
{
	pMatrix[0][0] = 1.0f;
	pMatrix[0][1] = 0.0f;
	pMatrix[0][2] = 0.0f;
	pMatrix[0][3] = 0.0f;
	pMatrix[1][0] = 0.0f;
	pMatrix[1][1] = 1.0f;
	pMatrix[1][2] = 0.0f;
	pMatrix[1][3] = 0.0f;
	pMatrix[2][0] = 0.0f;
	pMatrix[2][1] = 0.0f;
	pMatrix[2][2] = 1.0f;
	pMatrix[2][3] = 0.0f;
	pMatrix[3][0] = 0.0f;
	pMatrix[3][1] = 0.0f;
	pMatrix[3][2] = 0.0f;
	pMatrix[3][3] = 1.0f;
}

void demo_matrixMutiply(SVGFloat psRes[4][4], SVGFloat psSrcA[4][4], SVGFloat psSrcB[4][4])
{
	SVGFloat fB00, fB01, fB02, fB03;
	SVGFloat fB10, fB11, fB12, fB13;
	SVGFloat fB20, fB21, fB22, fB23;
	SVGFloat fB30, fB31, fB32, fB33;
	SVGUint32 i;

	fB00 = psSrcB[0][0]; fB01 = psSrcB[0][1]; fB02 = psSrcB[0][2]; fB03 = psSrcB[0][3];
	fB10 = psSrcB[1][0]; fB11 = psSrcB[1][1]; fB12 = psSrcB[1][2]; fB13 = psSrcB[1][3];
	fB20 = psSrcB[2][0]; fB21 = psSrcB[2][1]; fB22 = psSrcB[2][2]; fB23 = psSrcB[2][3];
	fB30 = psSrcB[3][0]; fB31 = psSrcB[3][1]; fB32 = psSrcB[3][2]; fB33 = psSrcB[3][3];

	for (i = 0; i < 4; i++)
	{
		psRes[i][0] = psSrcA[i][0]*fB00 + psSrcA[i][1]*fB10	+ psSrcA[i][2]*fB20 + psSrcA[i][3]*fB30;
		psRes[i][1] = psSrcA[i][0]*fB01 + psSrcA[i][1]*fB11	+ psSrcA[i][2]*fB21 + psSrcA[i][3]*fB31;
		psRes[i][2] = psSrcA[i][0]*fB02 + psSrcA[i][1]*fB12	+ psSrcA[i][2]*fB22 + psSrcA[i][3]*fB32;
		psRes[i][3] = psSrcA[i][0]*fB03 + psSrcA[i][1]*fB13	+ psSrcA[i][2]*fB23 + psSrcA[i][3]*fB33;
	}
}

void demo_matrixTranslate(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY)
{
	pMatrix[3][0] += fX * pMatrix[0][0] + fY * pMatrix[1][0];
	pMatrix[3][1] += fX * pMatrix[0][1] + fY * pMatrix[1][1];
	pMatrix[3][2] += fX * pMatrix[0][2] + fY * pMatrix[1][2];
	pMatrix[3][3] += fX * pMatrix[0][3] + fY * pMatrix[1][3];
}

void demo_matrixIdentity(SVGFloat pMatrix[4][4])
{
	pMatrix[0][0] = 1.0f;
	pMatrix[0][1] = 0.0f;
	pMatrix[0][2] = 0.0f;
	pMatrix[0][3] = 0.0f;
	pMatrix[1][0] = 0.0f;
	pMatrix[1][1] = 1.0f;
	pMatrix[1][2] = 0.0f;
	pMatrix[1][3] = 0.0f;
	pMatrix[2][0] = 0.0f;
	pMatrix[2][1] = 0.0f;
	pMatrix[2][2] = 1.0f;
	pMatrix[2][3] = 0.0f;
	pMatrix[3][0] = 0.0f;
	pMatrix[3][1] = 0.0f;
	pMatrix[3][2] = 0.0f;
	pMatrix[3][3] = 1.0f;
}

void local_matrixMutiply(SVGFloat psRes[4][4], SVGFloat psSrcA[4][4], SVGFloat psSrcB[4][4])
{
	SVGFloat fB00, fB01, fB02, fB03;
	SVGFloat fB10, fB11, fB12, fB13;
	SVGFloat fB20, fB21, fB22, fB23;
	SVGFloat fB30, fB31, fB32, fB33;
	SVGUint32 i;

	fB00 = psSrcB[0][0]; fB01 = psSrcB[0][1]; fB02 = psSrcB[0][2]; fB03 = psSrcB[0][3];
	fB10 = psSrcB[1][0]; fB11 = psSrcB[1][1]; fB12 = psSrcB[1][2]; fB13 = psSrcB[1][3];
	fB20 = psSrcB[2][0]; fB21 = psSrcB[2][1]; fB22 = psSrcB[2][2]; fB23 = psSrcB[2][3];
	fB30 = psSrcB[3][0]; fB31 = psSrcB[3][1]; fB32 = psSrcB[3][2]; fB33 = psSrcB[3][3];

	for (i = 0; i < 4; i++)
	{
		psRes[i][0] = psSrcA[i][0]*fB00 + psSrcA[i][1]*fB10	+ psSrcA[i][2]*fB20 + psSrcA[i][3]*fB30;
		psRes[i][1] = psSrcA[i][0]*fB01 + psSrcA[i][1]*fB11	+ psSrcA[i][2]*fB21 + psSrcA[i][3]*fB31;
		psRes[i][2] = psSrcA[i][0]*fB02 + psSrcA[i][1]*fB12	+ psSrcA[i][2]*fB22 + psSrcA[i][3]*fB32;
		psRes[i][3] = psSrcA[i][0]*fB03 + psSrcA[i][1]*fB13	+ psSrcA[i][2]*fB23 + psSrcA[i][3]*fB33;
	}
}


void init(void)
{

    GLuint i;
    char pszInfoLog[1024];
    SVGInt32  nShaderStatus, nInfoLogLength;
    SVGInt32 hShaderHandle[2];

    /* Create a program object */
    hShaderHandle[0] = glCreateShader(GL_VERTEX_SHADER);
    hShaderHandle[1] = glCreateShader(GL_FRAGMENT_SHADER);
    hProgramHandle = glCreateProgram();

    SVGInt32  nProgramLength;
    for(i=0; i<2 ; i++)
    {
        nProgramLength = strlen(apszFiles[i]);
        glShaderSource(hShaderHandle[i], 1, (const char **)&apszFiles[i], &nProgramLength);
        /* Compile the shader */
        glCompileShader(hShaderHandle[i]);
        /* Check it compiled OK */
        glGetShaderiv(hShaderHandle[i], GL_COMPILE_STATUS, &nShaderStatus);
        if (nShaderStatus != GL_TRUE)
        {
            puts("Error: Failed to compile GLSL shader");
            glGetShaderInfoLog(hShaderHandle[i], 1024, &nInfoLogLength, pszInfoLog);
            printf("%s",pszInfoLog);
            return ;
        }
    }
    /* Attach the shader to the programs */
    glAttachShader(hProgramHandle, hShaderHandle[0]);
    glAttachShader(hProgramHandle, hShaderHandle[1]);


    glBindAttribLocation(hProgramHandle, 0, "position");

    /* Link the program */
    glLinkProgram(hProgramHandle);

    /* Check it linked OK */
    glGetProgramiv(hProgramHandle, GL_LINK_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        puts("Error: Failed to link GLSL program");
        glGetProgramInfoLog(hProgramHandle, 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return ;
    }

    glValidateProgram(hProgramHandle);

    glGetProgramiv(hProgramHandle, GL_VALIDATE_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        puts("Error: Failed to validate GLSL program");
        glGetProgramInfoLog(hProgramHandle, 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return;
    }

    glUniform1i(glGetUniformLocation(hProgramHandle, "basetexture"), 0);

    texturelocation = glGetAttribLocation(hProgramHandle, "inputtexcoord");
    vertexlocation  = glGetAttribLocation(hProgramHandle, "position");

    /* end of glyph shader initialisation */

    glUseProgram(hProgramHandle);

    mvplocation = glGetUniformLocation(hProgramHandle, "mvp");
    glUniformMatrix4fv(mvplocation, 1, GL_FALSE, &projection[0][0]);
    glEnableVertexAttribArray (0);

    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

    return;
}



typedef struct tim_
{
    struct timeval srt;
    struct timeval end;
} TIM;

void time_start(TIM *tim)
{
    gettimeofday(&tim->srt, NULL );
}

unsigned int time_end(TIM *tim)
{
    gettimeofday(&tim->end, NULL );
    return (tim->end.tv_sec - tim->srt.tv_sec) * 1000 + (tim->end.tv_usec
            - tim->srt.tv_usec) / 1000;
}

/* Function definitions */


static SVGInt32 tst_bmpdec_read_img(const SVGChar* p_filename,
        SVGUint8* p_imagedata)
{
    FILE *fpr;
    struct stat attr;
    SVGInt32 ret;


    if (p_imagedata == NULL)
    {
        printf("ERROR in %s [null pointer]\n", __FUNCTION__);
        return 0;
    }

    stat(p_filename, &attr);

    fpr = fopen(p_filename, "rb");
    if (fpr == NULL)
    {

        printf("ERROR in %s [CANNOT OPEN FILE] %s \n", __FUNCTION__, p_filename);
        return 0;
    }


    ret = (SVGInt32) fread(p_imagedata, 1, (SVGUint32) attr.st_size, fpr);
    fclose(fpr);
    return ret;
}

static void tst_bmpdec_draw_image(SVGInt16* pos_x, SVGInt16* pos_y, SVGImage* svg_image)
{
	SVGUint32 handle   		= 0;
	SVGUint32 cnt_line 		= 0;
	SVGUint8* Resource 		= 0;
    SVGUint32 a_width      	= svg_image->width;
    SVGUint32 a_height     	= svg_image->height;
    SVGUint32 width         = svg_image->clip.width;
    SVGUint32 height        = svg_image->clip.height;
	GLfloat   vertx[8]     	= {0};
	GLfloat   texCoords[8]  = {0};

    projection[0][0] = 2.0f / SCREEN_WIDTH;
    projection[1][1] = 2.0f / SCREEN_HEIGHT;

	glEnableVertexAttribArray (vertexlocation);
    glEnableVertexAttribArray (texturelocation);
    local_matrixIdentity(modelview);
    local_matrixMutiply(mvp, modelview, projection);
    glUniformMatrix4fv(mvplocation, 1, GL_FALSE, &mvp[0][0]);

    glGenTextures(1, &handle);
	glBindTexture(GL_TEXTURE_2D, handle);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	if((a_width &(a_width-1)) != 0)
	{
		a_width--;
		a_width |= (a_width >> 1);
		a_width |= (a_width >> 2);
		a_width |= (a_width >> 4);
		a_width |= (a_width >> 8);
		a_width |= (a_width >> 16);
		a_width++;
	}

	if((a_height &(a_height-1)) != 0)
	{
		a_height--;
		a_height |= (a_height >> 1);
		a_height |= (a_height >> 2);
		a_height |= (a_height >> 4);
		a_height |= (a_height >> 8);
		a_height |= (a_height >> 16);
		a_height++;
	}
	if((width == 0))
	{
		width = svg_image->width;
	}
	if((height == 0))
	{
		height = svg_image->height;
	}

	// clear the texture
	Resource = (SVGUint8*)malloc(a_height * a_width * 4);
	if(Resource){
	memset(Resource, 0xff, a_height * a_width * 4);
	}
	glTexImage2D(GL_TEXTURE_2D,
			 0,
			 GL_RGBA,
			 (GLsizei)(a_width),
			 (GLsizei)(a_height),
			 0,
			 GL_RGBA,
             GL_UNSIGNED_BYTE,
			 Resource
			 );
	free(Resource);

	for (cnt_line = 0; cnt_line < height ; cnt_line++)
	{
		// fill the texture linewise, because it is not power-of-two-aligned
		glTexSubImage2D(GL_TEXTURE_2D,
						0,
						(GLint)(0),
						(GLint)((height -1) - cnt_line),
						(GLsizei)width,
						1,
						GL_RGBA,
	                    GL_UNSIGNED_BYTE,
						&svg_image->dest.dest_mem.dest_mem[cnt_line * svg_image->width * 4]);
	}

    /* just for positioning */
	if((*pos_x == 0) && (*pos_y == 0))
	{
		max_height = 0;
	}
	if(*pos_x +svg_image->width > SCREEN_WIDTH)
	{
		if(*pos_x > 0)
		{
			*pos_x = 0;
			*pos_y += max_height;
			max_height = 0;
		}
	}
	if(svg_image->height > max_height)
	{
		max_height = svg_image->height;
	}

	vertx[0] = 0 /*+ *pos_x */ + svg_image->clip.x;
	vertx[1] = 0 /*+ *pos_y */ + svg_image->clip.y;
	vertx[2] = vertx[0] + ((width));
	vertx[3] = vertx[1];
	vertx[4] = vertx[2];
	vertx[5] = vertx[1] + ((height));
	vertx[6] = vertx[0];
	vertx[7] = vertx[5];

	texCoords[0] = 0;
	texCoords[1] = 0;
	texCoords[2] = ( (SVGFloat)width / (SVGFloat)a_width);
	texCoords[3] = texCoords[1];
	texCoords[4] = texCoords[2];
	texCoords[5] = ( (SVGFloat)height / (SVGFloat)a_height);
	texCoords[6] = texCoords[0];
	texCoords[7] = texCoords[5];

	glVertexAttribPointer(texturelocation, 2, GL_FLOAT, 0, 0, texCoords);
	glVertexAttribPointer(vertexlocation, 2, GL_FLOAT, 0, 0, vertx);

	glClear(GL_COLOR_BUFFER_BIT);

	glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
    eglSwapBuffers(eglDisplay, eglWindowSurface);
#ifdef USE_WAYLAND
        WaylandGles* sink = &aWayland;
        struct wl_callback* callback = wl_surface_frame(sink->wl_surface);
        wl_callback_add_listener(callback, &framelistener, NULL);
        wl_display_flush(sink->wl_display);
#endif

    usleep(1000000);
	glDeleteTextures(1, &handle);
	*pos_x += svg_image->width;
}


static SVGInt32 tst_bmpdec_png_queued(void)
{
    SVGInt32 num_fail = 0;
    SVGImage svg_image[14];
    SVGImageInfo svg_image_info;
    TIM time_q;
    SVGInt32 dec_time = 0;
    SVGInt32 cnt_img = 0;
    SVGUint32 bytes = 0;
    SVGError err = 0;
    SVGChar* filename[14] =
    { (SVGChar*) (void*) "./images/BASI0G16.PNG",
 	  (SVGChar*) (void*) "./images/BASI2C16.PNG",
	  (SVGChar*) (void*) "./images/BASI4A16.PNG",
	  (SVGChar*) (void*) "./images/BASI6A16.PNG",
	  (SVGChar*) (void*) "./images/BASN0G01.PNG",
	  (SVGChar*) (void*) "./images/BASN0G02.PNG",
	  (SVGChar*) (void*) "./images/BASN0G04.PNG",
	  (SVGChar*) (void*) "./images/BASN0G08.PNG",
	  (SVGChar*) (void*) "./images/BASN0G16.PNG",
	  (SVGChar*) (void*) "./images/BASN2C16.PNG",
	  (SVGChar*) (void*) "./images/BASN4A16.PNG",
	  (SVGChar*) (void*) "./images/BASN6A16.PNG",
	  (SVGChar*) (void*) "./images/BASN3P08.PNG" };

    SVGInt16 dst_pos_x = 0;
    SVGInt16 dst_pos_y = 0;
    SVGInt16 dst_clip_x = 0;
    SVGInt16 dst_clip_y = 0;
    SVGInt16 dst_clip_width = 0;
    SVGInt16 dst_clip_height = 0;
    SVGUint8* p_img_buf[14];
    SVGUint8* arr_svg_surf_dec[14] = {NULL};

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Testing PNG decoding with queued requests..."));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "--------------------------------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    dst_pos_x = 0;
    dst_pos_y = 10 * 33;

    /* Normal position */
    dst_clip_x = 0;
    dst_clip_y = 0;
    dst_clip_width = SMALL_CLIP;
    dst_clip_height = SMALL_CLIP;


    for (cnt_img = 0; cnt_img < 13; cnt_img++)
    {

        /* Load image from FS */
        p_img_buf[cnt_img] = malloc(PNG_MAX_SIZE_SMALL);
        if (NULL != p_img_buf[cnt_img])
        {
            bytes = tst_bmpdec_read_img(filename[cnt_img], p_img_buf[cnt_img]);
            svg_image[cnt_img].datap = (SVGInt8*) (void*) p_img_buf[cnt_img];

            /* Configure Decoder */
            svg_image[cnt_img].type = SVG_BMPDEC_PNG_ENCODING;
            svg_image[cnt_img].data_size = bytes;


            svgGetImageInfo(gp_bmp_ctx, &svg_image[cnt_img], &svg_image_info);
            err = svgGetBmpDecError(gp_bmp_ctx);
            if (SVG_NO_ERROR != err)
            {
                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while getting info!"));
                num_fail++;
            } else
            {

                /* Create surface for decoding */
                arr_svg_surf_dec[cnt_img] = malloc(svg_image_info.width* svg_image_info.height *4);


                if(NULL != arr_svg_surf_dec[cnt_img])
                {
					/* Set decoding parameters */
					svg_image[cnt_img].decode_mode = SVG_BMPDEC_MODE_MEMORY;
					svg_image[cnt_img].dest.dest_mem.dest_mem = (SVGUint8*)arr_svg_surf_dec[cnt_img];
					svg_image[cnt_img].dest.dest_mem.dest_size
							= svg_image_info.width * svg_image_info.height * 4;
					svg_image[cnt_img].width = svg_image_info.width;
					svg_image[cnt_img].height = svg_image_info.height;
					svg_image[cnt_img].clip.x = dst_clip_x;
					svg_image[cnt_img].clip.y = dst_clip_y;
					svg_image[cnt_img].clip.width = dst_clip_width;
					svg_image[cnt_img].clip.height = dst_clip_height;

					/* Decode image */
					time_start(&time_q);
					svgDecodeImage(gp_bmp_ctx, &svg_image[cnt_img]);
					err = svgGetBmpDecError(gp_bmp_ctx);
					if (SVG_NO_ERROR != err)
					{
						//TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while queuing request!"));
						num_fail++;
					}
                }
            }
        } else
        {
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while allocating memory!"));
        }
    }

    /* Now that all decoding requests have been queued, trigger the actual decoding */
    svgFinishBmpDec(gp_bmp_ctx);
    err = svgGetBmpDecError(gp_bmp_ctx);
    if (SVG_NO_ERROR != err)
    {
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while decoding!"));
        num_fail++;
    } else
    {
        dec_time = time_end(&time_q);

        /* Copy all decoded surfaces to the screen */

        for (cnt_img = 0; cnt_img < 13; cnt_img++)
        {
        	if (NULL != arr_svg_surf_dec[cnt_img])
            {
                tst_bmpdec_draw_image(&dst_pos_x, &dst_pos_y, &svg_image[cnt_img]);
            }
        }

        sprintf(&string[0], " -> Average decoding time: %5dms", dec_time / 12);
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, string));
    }

    /* Free resources */
    for (cnt_img = 0; cnt_img < 13; cnt_img++)
    {
        free(arr_svg_surf_dec[cnt_img]);
        free(p_img_buf[cnt_img]);
    }

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}

#ifdef COMMENTED_OUT

static SVGInt32 tst_bmpdec_png_invalid(void)
{
    SVGInt32 num_fail = 0;
    SVGUint8* p_img_buf = NULL;
    SVGImage svg_image;
    SVGImageInfo svg_image_info;
    SVGUint32 bytes = 0;
    SVGError err = 0;
    SVGChar* filename = (SVGChar*) (void*) "./images/tst_no_plte.PNG";
    SVGInt16 dst_clip_x = 0;
    SVGInt16 dst_clip_y = 0;
    SVGInt16 dst_clip_width = 0;
    SVGInt16 dst_clip_height = 0;
    SVGUint8* arr_svg_surf_dec[1] = {NULL};

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Testing decoding an invalid PNG..."));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "-----------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    dst_clip_x = 0;
    dst_clip_y = 0;
    dst_clip_width = SMALL_CLIP;
    dst_clip_height = SMALL_CLIP;

    memset(&svg_image, 0, sizeof(svg_image));

    /* Load image from FS */
    p_img_buf = malloc(PNG_MAX_SIZE_SMALL);
    bytes = tst_bmpdec_read_img(filename, p_img_buf);
    svg_image.datap = (SVGInt8*) (void*) p_img_buf;

    /* Configure Decoder */
    svg_image.type = SVG_BMPDEC_PNG_ENCODING;
    svg_image.data_size = bytes;

    svgGetImageInfo(gp_bmp_ctx, &svg_image, &svg_image_info);
    err = svgGetBmpDecError(gp_bmp_ctx);
    if (SVG_BMPDEC_DEC_ERROR != err)
    {
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR! Invalid PNG did not result in error code when getting image info."));
        num_fail++;
    } else
    {
        /* Create surface for decoding */
    	arr_svg_surf_dec[0] = malloc(SMALL_CLIP * SMALL_CLIP * 4);

        /* Set decoding parameters */
        svg_image.decode_mode = SVG_BMPDEC_MODE_MEMORY;
        svg_image.dest.dest_mem.dest_mem = (SVGUint8*)arr_svg_surf_dec[0];
        svg_image.dest.dest_mem.dest_size =SMALL_CLIP * SMALL_CLIP * 4;

        svg_image.width = SMALL_CLIP;
        svg_image.height = SMALL_CLIP;
        svg_image.clip.x = dst_clip_x;
        svg_image.clip.y = dst_clip_y;
        svg_image.clip.width = dst_clip_width;
        svg_image.clip.height = dst_clip_height;

        /* Decode image to check if it also returns an error for the invalid PNG */
        svgDecodeImage(gp_bmp_ctx, &svg_image);
        err = svgGetBmpDecError(gp_bmp_ctx);

        svgFinishBmpDec(gp_bmp_ctx);
        err = svgGetBmpDecError(gp_bmp_ctx);
        if (SVG_BMPDEC_DEC_ERROR != err)
        {
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR! Invalid PNG did not result in error code when decoding."));
            num_fail++;
        }
    }
    /* Free resources */
    free(arr_svg_surf_dec[0]);
    free(p_img_buf);

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}
#endif

static SVGInt32 tst_bmpdec_formats(void)
{
    SVGInt32 num_fail = 0;
    SVGBmpDecFormats formats = 0;
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Checking supported formats..."));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "-----------------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    svgGetBmpDecFormats(gp_bmp_ctx, &formats);

    if (0 != (formats & SVG_BMPDEC_FORMAT_JPEG))
    {
        /* JPEG is supported */
        printf("SVG_BMPDEC_FORMAT_JPEG format: %d\n",SVG_BMPDEC_FORMAT_JPEG);
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> JPEG is supported"));
    }
    else
    {
        num_fail++;
    }

    if (0 != (formats & SVG_BMPDEC_FORMAT_PNG))
    {
        /* PNG is supported */
        printf("SVG_BMPDEC_FORMAT_PNG format: %d\n",SVG_BMPDEC_FORMAT_PNG);
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> PNG is supported"));
    }
    else
    {
        num_fail++;
    }

    if (0 != (formats & SVG_BMPDEC_FORMAT_FGS))
    {
        /* PNG is supported */
        printf("SVG_BMPDEC_FORMAT_FGS format: %d\n",SVG_BMPDEC_FORMAT_FGS);
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> PNG is supported"));
    }
    else
    {
        num_fail++;
    }
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}

static SVGInt32 tst_bmpdec_errors(void)
{
    SVGInt32 num_fail = 0;
    SVGError err = SVG_NO_ERROR;
    SVGUint8* p_img_buf = NULL;
    SVGImage svg_image;
    SVGImageInfo svg_image_info;
    SVGBmpDecFormats formats = 0;
    SVGUint32 bytes = 0;
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Testing some error conditions"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "-----------------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    /****************************************************/
    /* Try to produce some error messages */
    svgDestroyContextBmpDec(gp_bmp_ctx);

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Initialize a second time..."));
    svgInitBmpDec();

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Shutdown uninitialized BmpDec..."));
    svgCloseBmpDec();
    svgCloseBmpDec();

    /****************************************************/
    /* Tests with uninitialized BmpDec context */

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Create context on uninitialized BmpDec..."));
    gp_bmp_ctx = svgCreateContextBmpDec();

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Destroy unexisting context on uninitialized BmpDec..."));
    svgDestroyContextBmpDec(gp_bmp_ctx);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Get image info on uninitialized BmpDec..."));
    memset(&svg_image, 0, sizeof(svg_image));
    svgGetImageInfo(gp_bmp_ctx, &svg_image, &svg_image_info);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Get supported formats on uninitialized BmpDec..."));
    svgGetBmpDecFormats(gp_bmp_ctx, &formats);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Decode image on uninitialized BmpDec..."));
    svgDecodeImage(gp_bmp_ctx, &svg_image);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Flush BmpDec on uninitialized BmpDec..."));
    svgFlushBmpDec(gp_bmp_ctx);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Finish BmpDec on uninitialized BmpDec..."));
    svgFinishBmpDec(gp_bmp_ctx);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Get error on uninitialized BmpDec..."));
    svgGetBmpDecError(gp_bmp_ctx);

    /****************************************************/
    /* Tests with initialized BmpDec context */

    svgInitBmpDec();
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Destroy NULL context..."));
    svgDestroyContextBmpDec(NULL);

    /****************************************************/
    gp_bmp_ctx = svgCreateContextBmpDec();
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Get image info with invalid params..."));
    svgGetImageInfo(NULL, NULL, NULL );
    svgGetImageInfo(gp_bmp_ctx, NULL, NULL );
    svgGetImageInfo(gp_bmp_ctx, &svg_image, NULL);

    /****************************************************/
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Decode image with invalid target surface..."));
    memset(&svg_image, 0, sizeof(svg_image));

    /* Load image from FS */
    p_img_buf = malloc(PNG_MAX_SIZE_SMALL);
    bytes = tst_bmpdec_read_img("./images/BASN0G02.PNG", p_img_buf);
    svg_image.datap = (SVGInt8*) (void*) p_img_buf;

    /* Configure Decoder */
    svg_image.type = SVG_BMPDEC_PNG_ENCODING;
    svg_image.data_size = bytes;

    svgGetImageInfo(gp_bmp_ctx, &svg_image, &svg_image_info);

    /* Set decoding parameters */
    svg_image.decode_mode = SVG_BMPDEC_MODE_MEMORY;
    svg_image.dest.dest_mem.dest_mem = NULL;
    svg_image.dest.dest_mem.dest_size = 0;
    svg_image.width = 50;
    svg_image.height = 0;
    svg_image.clip.x = 0;
    svg_image.clip.y = 0;
    svg_image.clip.width = svg_image_info.width;
    svg_image.clip.height = svg_image_info.height;

    /* Decode image */
    svgDecodeImage(gp_bmp_ctx, &svg_image);
    err = svgGetBmpDecError(gp_bmp_ctx);
    if (SVG_INVALID_VALUE != err)
    {
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> Returned correct error code"));
    }

    free(p_img_buf);

    /* Remove any remaining error */
    svgGetBmpDecError(gp_bmp_ctx);

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}

static SVGInt32 tst_bmpdec_mem_dest(void)
{
    SVGInt32 num_fail = 0;
    SVGUint8* p_img_buf = NULL;
    SVGUint8* p_dec_img_buf = NULL;
    SVGImage svg_image;
    SVGImageInfo svg_image_info;
    TIM time_mem;
    SVGInt32 dec_time = 0;
    SVGInt32 total_time = 0;
    SVGInt32 cnt_img = 0;
    SVGUint32 bytes = 0;
    SVGError err = 0;
    SVGChar* filename[36] =
    { (SVGChar*) (void*) "./images/BASI0G16.JPG",
	  (SVGChar*) (void*) "./images/BASI2C16.JPG",
	  (SVGChar*) (void*) "./images/BASI4A16.JPG",
	  (SVGChar*) (void*) "./images/BASI6A16.JPG",
	  (SVGChar*) (void*) "./images/BASN0G01.JPG",
	  (SVGChar*) (void*) "./images/BASN0G02.JPG",
	  (SVGChar*) (void*) "./images/BASN0G04.JPG",
	  (SVGChar*) (void*) "./images/BASN0G08.JPG",
	  (SVGChar*) (void*) "./images/BASN0G16.JPG",
	  (SVGChar*) (void*) "./images/BASN2C16.JPG",
	  (SVGChar*) (void*) "./images/BASN4A16.JPG",
	  (SVGChar*) (void*) "./images/BASN6A16.JPG",
	  (SVGChar*) (void*) "./images/BASI0G16.JPG",
          (SVGChar*) (void*) "./images/lake_prog800_480.jpeg",
	  (SVGChar*) (void*) "./images/Devilsspit.JPG",
	  (SVGChar*) (void*) "./images/Right-top.JPG",
	  (SVGChar*) (void*) "./images/test_full_screen.fgs",
	  (SVGChar*) (void*) "./images/test_icon.fgs",
	  (SVGChar*) (void*) "./images/test_invalid.fgs",
	  (SVGChar*) (void*) "./images/BASN2C16.PNG",
	  (SVGChar*) (void*) "./images/img_800x600.PNG",
	  (SVGChar*) (void*) "./images/BASI0G16.PNG",
	  (SVGChar*) (void*) "./images/BASI0G16.PNG",
	  (SVGChar*) (void*) "./images/BASI2C16.PNG",
	  (SVGChar*) (void*) "./images/BASI4A16.PNG",
	  (SVGChar*) (void*) "./images/BASI6A16.PNG",
	  (SVGChar*) (void*) "./images/BASN0G01.PNG",
	  (SVGChar*) (void*) "./images/BASN0G02.PNG",
	  (SVGChar*) (void*) "./images/BASN0G04.PNG",
	  (SVGChar*) (void*) "./images/BASN0G08.PNG",
	  (SVGChar*) (void*) "./images/BASN0G16.PNG",
	  (SVGChar*) (void*) "./images/BASN4A16.PNG",
	  (SVGChar*) (void*) "./images/BASN6A16.PNG",
	  (SVGChar*) (void*) "./images/BASN3P08.PNG",
	  (SVGChar*) (void*) "./images/BASN3P08.PLTE"};

    SVGInt16 dst_pos_x = 0;
    SVGInt16 dst_pos_y = 0;
    SVGUint32 cnt_pos = 0;
    SVGInt16 dst_clip_x = 0;
    SVGInt16 dst_clip_y = 0;
    SVGInt16 dst_clip_width = SMALL_CLIP;
    SVGInt16 dst_clip_height = SMALL_CLIP;
    SVGUint8* p_plte_data = NULL;

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Testing decoding to memory destination..."));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "-----------------------------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    dst_pos_x = 0;
    dst_pos_y = 0;

    for (cnt_pos = 0; cnt_pos < NUM_POS; cnt_pos++)
    {
        /* Each iteration with different position and clipping settings */
        switch (cnt_pos)
        {
        case 0:
        {
            /* Normal position */
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " Normal position"));
            dst_clip_x = 0;
            dst_clip_y = 0;
            dst_clip_width = 0;
            dst_clip_height = 0;
            break;
        }
        case 1:
        {
            /* Left top */
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " Left top"));
            dst_clip_x = 0;
            dst_clip_y = 0;
            dst_clip_width = HALF_SMALL_CLIP;
            dst_clip_height = HALF_SMALL_CLIP;
            break;
        }
        case 2:
        {
            /* Bottom right */
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " Bottom right"));
            dst_clip_x = 0;
            dst_clip_y = HALF_SMALL_CLIP;
            dst_clip_width = HALF_SMALL_CLIP;
            dst_clip_height = HALF_SMALL_CLIP;
            break;
        }
        case 3:
        {
            /* Clip top left */
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " Clip top left"));
            dst_clip_x = HALF_SMALL_CLIP;
            dst_clip_y = 0;
            dst_clip_width = HALF_SMALL_CLIP;
            dst_clip_height = HALF_SMALL_CLIP;
            break;
        }
        case 4:
        {
            /* Clip bottom right */
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " Clip top right"));
            dst_clip_x = HALF_SMALL_CLIP;
            dst_clip_y = HALF_SMALL_CLIP;
            dst_clip_width = HALF_SMALL_CLIP;
            dst_clip_height = HALF_SMALL_CLIP;
            break;
        }
        default:
        {
        }
        }

        for (cnt_img = 0; cnt_img < 34; cnt_img++)
        {

            /* Load image from FS */
            p_img_buf = malloc(PNG_MAX_SIZE_SMALL);
            if (NULL != p_img_buf)
            {
                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> Memory allocation OK"));
                bytes = tst_bmpdec_read_img(filename[cnt_img], p_img_buf);
                svg_image.datap = (SVGInt8*) p_img_buf;

                /* Configure Decoder */
                svg_image.type = SVG_BMPDEC_JPEG_ENCODING;
                if (cnt_img > 15)
                {
            		svg_image.type = SVG_BMPDEC_FGS_ENCODING;
                }
            	if(cnt_img > 18)
            	{
            		svg_image.type = SVG_BMPDEC_PNG_ENCODING;
            	}

                svg_image.data_size = bytes;

                svgGetImageInfo(gp_bmp_ctx, &svg_image, &svg_image_info);
                err = svgGetBmpDecError(gp_bmp_ctx);
                if (SVG_NO_ERROR != err)
                {
                    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while getting info!"));
                    if( cnt_img != 18) /*add exception for ./images/test_invalid.fgs*/
                        num_fail++;
                } else
                {
                    /* Load PLTE for one test */
                    if (33 == cnt_img)
                    {
                        p_plte_data = malloc(3 * 256 + 4 + 4); /* palette + header + crc */
                        bytes = tst_bmpdec_read_img(filename[cnt_img + 1],
                                p_plte_data);
                        svg_image.type = SVG_BMPDEC_PNG_ENCODING_WITH_PLTE;
                        svg_image.attribute.png.plte_chunk_size = bytes;
                        svg_image.attribute.png.plte_chunk
                                = (SVGInt8*) p_plte_data;
                        svg_image.attribute.png.trns_chunk = NULL;
                        svg_image.attribute.png.trns_chunk_size = 0;
                    }

                    /* Allocate DRAM block for decoding */
                    p_dec_img_buf = (SVGUint8*) malloc(svg_image_info.width
                            * svg_image_info.height * 4);
                    if (NULL != p_dec_img_buf)
                    {
                        memset(p_dec_img_buf, 0x0, (svg_image_info.width
                                * svg_image_info.height * 4));

                        /* Set decoding parameters */
                        svg_image.decode_mode = SVG_BMPDEC_MODE_MEMORY;
                        svg_image.dest.dest_mem.dest_mem = p_dec_img_buf;
                        svg_image.dest.dest_mem.dest_size
                                = svg_image_info.width * svg_image_info.height
                                        * 4;
                        svg_image.width = svg_image_info.width;
                        svg_image.height = svg_image_info.height;
                        svg_image.clip.x = dst_clip_x;
                        svg_image.clip.y = dst_clip_y;
                        svg_image.clip.width = dst_clip_width;
                        svg_image.clip.height = dst_clip_height;

                        /* Decode image */
                        time_start(&time_mem);
                        svgDecodeImage(gp_bmp_ctx, &svg_image);
                        err = svgGetBmpDecError(gp_bmp_ctx);
                        if (SVG_NO_ERROR != err)
                        {
                            printf(" -> ERROR while queuing request!\n");
                            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while queuing request!"));
                            num_fail++;
                        } else
                        {
                            svgFinishBmpDec(gp_bmp_ctx);
                            err = svgGetBmpDecError(gp_bmp_ctx);
                            if (SVG_NO_ERROR != err)
                            {
                                printf(" -> ERROR while decoding!\n");
                                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while decoding!"));
                                num_fail++;
                            } else
                            {
                                dec_time = time_end(&time_mem);
                                tst_bmpdec_draw_image(&dst_pos_x, &dst_pos_y, &svg_image);

                                total_time += dec_time;

                            }
                        }
                        free(p_dec_img_buf);
                    } else
                    {
                        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> Error allocating destination memory!"));
                    }
                }
                /* Free resources */
                free(p_img_buf);

                sprintf(&string[0], " -> Average decoding time: %5dms",
                        dec_time);
                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, string));
                total_time = 0;
            } else
            {
                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> Error allocating memory!"));
            }

        }
    }

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}




static SVGInt32 tst_bmpdec_async(void)
{
    SVGInt32 num_fail = 0;
    SVGImage svg_image[14];
    SVGImageInfo svg_image_info;
    TIM time_asy;
    TIM flush_tm;
    SVGInt32 dec_time = 0;
    SVGInt32 flush_time = 0;
    SVGInt32 cnt_img = 0;
    SVGUint32 bytes = 0;
    SVGError err = 0;
    SVGChar* filename[14] =
    { (SVGChar*) (void*) "./images/BASI0G16.PNG",
            (SVGChar*) (void*) "./images/BASI2C16.PNG",
            (SVGChar*) (void*) "./images/BASI4A16.PNG",
            (SVGChar*) (void*) "./images/BASI6A16.PNG",
            (SVGChar*) (void*) "./images/BASN0G01.PNG",
            (SVGChar*) (void*) "./images/BASN0G02.PNG",
            (SVGChar*) (void*) "./images/BASN0G04.PNG",
            (SVGChar*) (void*) "./images/BASN0G08.PNG",
            (SVGChar*) (void*) "./images/BASN0G16.PNG",
            (SVGChar*) (void*) "./images/BASN2C16.PNG",
            (SVGChar*) (void*) "./images/BASN4A16.PNG",
            (SVGChar*) (void*) "./images/BASN6A16.PNG",
            (SVGChar*) (void*) "./images/BASN3P08.PNG" };

    SVGInt16 dst_pos_x = 0;
    SVGInt16 dst_pos_y = 0;
    SVGInt16 dst_clip_x = 0;
    SVGInt16 dst_clip_y = 0;
    SVGInt16 dst_clip_width = 0;
    SVGInt16 dst_clip_height = 0;
    SVGUint8* arr_svg_surf_dec[14] = {NULL};
    SVGUint8* p_img_buf[14];

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "Testing asynchronous decoding with queued requests..."));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, "-----------------------------------------------------"));
    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));

    dst_pos_x = 0;
    dst_pos_y = 10 * 33;

    /* Normal position */
    dst_clip_x = 0;
    dst_clip_y = 0;
    dst_clip_width = SMALL_CLIP;
    dst_clip_height = SMALL_CLIP;


    for (cnt_img = 0; cnt_img < 13; cnt_img++)
    {

        /* Load image from FS */
        p_img_buf[cnt_img] = malloc(PNG_MAX_SIZE_SMALL);
        if (NULL != p_img_buf[cnt_img])
        {
            bytes = tst_bmpdec_read_img(filename[cnt_img], p_img_buf[cnt_img]);
            svg_image[cnt_img].datap = (SVGInt8*) (void*) p_img_buf[cnt_img];

            /* Configure Decoder */
            svg_image[cnt_img].type = SVG_BMPDEC_PNG_ENCODING;
            svg_image[cnt_img].data_size = bytes;

            svgGetImageInfo(gp_bmp_ctx, &svg_image[cnt_img], &svg_image_info);
            err = svgGetBmpDecError(gp_bmp_ctx);
            if (SVG_NO_ERROR != err)
            {
                //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while getting info!"));
                num_fail++;
            } else
            {

                /* Create surface for decoding */
                arr_svg_surf_dec[cnt_img] = malloc(svg_image_info.width * svg_image_info.height *4);

                /* Set decoding parameters */
                svg_image[cnt_img].decode_mode = SVG_BMPDEC_MODE_MEMORY;
                svg_image[cnt_img].dest.dest_mem.dest_mem = (SVGUint8*)arr_svg_surf_dec[cnt_img];
                svg_image[cnt_img].dest.dest_mem.dest_size = svg_image_info.width * svg_image_info.height *4;
                svg_image[cnt_img].width = svg_image_info.width;
                svg_image[cnt_img].height = svg_image_info.height;
                svg_image[cnt_img].clip.x = dst_clip_x;
                svg_image[cnt_img].clip.y = dst_clip_y;
                svg_image[cnt_img].clip.width = dst_clip_width;
                svg_image[cnt_img].clip.height = dst_clip_height;

                /* Decode image */
                time_start(&time_asy);
                time_start(&flush_tm);
                svgDecodeImage(gp_bmp_ctx, &svg_image[cnt_img]);
                err = svgGetBmpDecError(gp_bmp_ctx);
                if (SVG_NO_ERROR != err)
                {
                    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while queuing request!"));
                    num_fail++;
                }
            }
        } else
        {
            //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR Could not allocate memory!"));
        }

    }

    /* Now that all decoding requests have been queued, trigger the actual decoding */
    svgFlushBmpDec(gp_bmp_ctx);
    err = svgGetBmpDecError(gp_bmp_ctx);
    if (SVG_NO_ERROR != err)
    {
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, " -> ERROR while decoding!"));
        num_fail++;
    }

    else
    {
        flush_time = time_end(&flush_tm);

        /* Try a second flush (should give an error trace) */
        // svgFlushBmpDec( gp_bmp_ctx );

        /* Wait for the decoder to finish */
        svgWaitBmpDec(gp_bmp_ctx);
        dec_time = time_end(&time_asy);

        for (cnt_img = 0; cnt_img < 13; cnt_img++)
        {
            tst_bmpdec_draw_image(&dst_pos_x, &dst_pos_y, &svg_image[cnt_img]);
        }
        sprintf(&string[0]," -> Average decoding time: %5dms     Flushing time: %5dms", dec_time / 12, flush_time );
        //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, string));
    }

    /* Free resources */
    for (cnt_img = 0; cnt_img < 13; cnt_img++)
    {
        if (NULL != arr_svg_surf_dec[cnt_img])
        {
            free(arr_svg_surf_dec[cnt_img]);
        }
        if (NULL != p_img_buf[cnt_img])
        {
            free(p_img_buf[cnt_img]);
        }
    }

    //TR_U3((TRACE_CLASS,"%2u%s", 0xffff, ""));
    return num_fail;
}


static void handle_egl_error(const char *name)
{
  static char * const error_strings[] =
    {
      "EGL_SUCCESS",
      "EGL_NOT_INITIALIZED",
      "EGL_BAD_ACCESS",
      "EGL_BAD_ALLOC",
      "EGL_BAD_ATTRIBUTE",
      "EGL_BAD_CONFIG",
      "EGL_BAD_CONTEXT",
      "EGL_BAD_CURRENT_SURFACE",
      "EGL_BAD_DISPLAY",
      "EGL_BAD_MATCH",
      "EGL_BAD_NATIVE_PIXMAP",
      "EGL_BAD_NATIVE_WINDOW",
      "EGL_BAD_PARAMETER",
      "EGL_BAD_SURFACE"
    };
  EGLint error_code=eglGetError();

  fprintf(stderr," %s: egl error \"%s\" (0x%x)\n", name,
          error_strings[error_code-EGL_SUCCESS], error_code);
  exit(EXIT_FAILURE);
}


static void eglOpen(int api)
{
	  EGLBoolean ret;
	  static const EGLint gl_context_attribs[] =
	    {
	      EGL_CONTEXT_CLIENT_VERSION, 2,
	      EGL_NONE
	    };

	   EGLint s_configAttribs[] =
	    {
	      EGL_RED_SIZE,	8,
	      EGL_GREEN_SIZE, 	8,
	      EGL_BLUE_SIZE,	8,
	      EGL_ALPHA_SIZE, 	EGL_DONT_CARE,
	      EGL_DEPTH_SIZE,         EGL_DONT_CARE,
	      EGL_STENCIL_SIZE,       EGL_DONT_CARE,
	      EGL_SURFACE_TYPE,           EGL_WINDOW_BIT,
	      EGL_RENDERABLE_TYPE,    EGL_OPENGL_ES2_BIT,
	      EGL_SAMPLES,            EGL_DONT_CARE,
	      EGL_NONE
	    };
	  EGLint numConfigs;
	  EGLint majorVersion;
	  EGLint minorVersion;

#ifndef USE_WAYLAND
  EGLNativeWindowType  native_window  = fbCreateWindow(fbGetDisplay(NULL), 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
#else
  /* set wayland environment to run without shell script */
  if(NULL == getenv("XDG_RUNTIME_DIR"))
  {
	  setenv("XDG_RUNTIME_DIR","/tmp/", TRUE);
  }
  WaylandGles* sink = &aWayland;
  sink->surface_id = 40;
  sink->layer_id = 3000;
  int adapter_error = 0;

  if (!create_wl_context(sink))
  {
	  printf("fail 2\n");
//	  eglClose();
	  return ;
  }

  eglDisplay = eglGetDisplay( (EGLNativeDisplayType) sink->wl_display);

  adapter_error = compositor_shim_surface_init (&sink->wlAdapterSurfaceContext, sink->wl_surface, sink->layer_id, 
                                                sink->surface_id, SCREEN_WIDTH, SCREEN_HEIGHT);
  if (adapter_error != 0)
  {
      printf("compositor_shim_surface_init() failed\n");
      return;
  }

  adapter_error = compositor_shim_surface_configure (sink->wlAdapterContext, &sink->wlAdapterSurfaceContext, ADAPTER_CONFIGURATION_ALL);
  if (adapter_error != 0)
  {
      printf("compositor_shim_surface_configure() failed\n");
      return;
  }
#endif /* USE_WAYLAND */

  if(api > 0)
  {
	  s_configAttribs[16] =  EGL_NONE;
    eglBindAPI(EGL_OPENGL_ES_API);
  }
  else
  {
    eglBindAPI(EGL_OPENVG_API);
  }
#ifndef USE_WAYLAND
  eglDisplay = eglGetDisplay(EGL_DEFAULT_DISPLAY);
#endif

  eglInitialize(eglDisplay, &majorVersion, &minorVersion);

  eglGetConfigs(eglDisplay, NULL, 0, &numConfigs);

  eglChooseConfig(eglDisplay, s_configAttribs, &eglConfig, 1, &numConfigs);
#ifndef USE_WAYLAND
  eglWindowSurface = eglCreateWindowSurface(eglDisplay, eglConfig, native_window, NULL);
#else
  eglWindowSurface = eglCreateWindowSurface(eglDisplay, eglConfig, (EGLNativeWindowType)sink->wl_native_window, NULL);
#endif
  if (eglWindowSurface == EGL_NO_SURFACE)
    {
      handle_egl_error("eglCreateWindowSurface");
    }

  if(api > 0)
  {
	  eglContext = eglCreateContext(eglDisplay, eglConfig, EGL_NO_CONTEXT, gl_context_attribs);
  }
  else
  {
	  eglContext = eglCreateContext(eglDisplay, eglConfig, EGL_NO_CONTEXT, NULL);
  }

  if (eglContext == EGL_NO_CONTEXT)
    {
      handle_egl_error("eglCreateContext");
    }

  ret = eglMakeCurrent(eglDisplay, eglWindowSurface, eglWindowSurface, eglContext);
  if (ret == EGL_FALSE)
    handle_egl_error("eglMakeCurrent");
  else
    printf("setup ok\n" );
  eglSwapInterval(eglDisplay,1);
}


static void eglClose(void)
{
      eglMakeCurrent(eglDisplay, NULL, NULL, NULL);
      eglDestroyContext(eglDisplay, eglContext);
      eglDestroySurface(eglDisplay, eglWindowSurface);
      eglTerminate(eglDisplay);
      eglReleaseThread();

#ifdef USE_WAYLAND
	  WaylandGles* sink = &aWayland;
          compositor_shim_terminate(sink->wlAdapterContext);

	  if (sink->wl_native_window)
	  {
	      wl_egl_window_destroy(sink->wl_native_window);
	  }

	  if (sink->wl_surface)
	  {
	      wl_surface_destroy(sink->wl_surface);
	  }

	  if (sink->wl_compositor)
	  {
	      wl_compositor_destroy(sink->wl_compositor);
	  }
#endif /* USE_WAYLAND */
}


int main(void)
{
    SVGInt32 num_fail = 0;
    SVGInt32 tst_fail = 0;
    SVGUint32 cnt_tst = 1;

    sprintf(&string[0], "Start unit tests (run %5d)", cnt_tst);

    eglOpen(1);
    init();

    svgInitBmpDec();

    svgGetBmpDecError(NULL);

    gp_bmp_ctx = svgCreateContextBmpDec();
    svgGetBmpDecError(gp_bmp_ctx);
    printf("INFO::[ Init OK ]\n");

    /* Test some error cases */
    printf("INFO::[ Testing BMPDEC ERRORS ]\n");
    tst_fail = tst_bmpdec_errors();
    num_fail += tst_fail;
    if (!num_fail)
        printf("INFO::[ BMPDEC ERRORS OK ]\n");
    else
        printf("INFO::[ BMPDEC ERRORS FAIL ]\n");
    /* Test Format list */
    printf("INFO::[ Testing BMPDEC FORMATS ]\n");
    tst_fail = tst_bmpdec_formats();
    num_fail += tst_fail;
    if (!num_fail)
        printf("INFO::[ BMPDEC FORMATS OK ]\n");
    else
        printf("INFO::[ BMPDEC FORMATS FAIL ]\n");


    /* Test PNG queued */
    printf("INFO::[ Testing BMPDEC PNG QUEUED ]\n");
    tst_fail = tst_bmpdec_png_queued();
    num_fail += tst_fail;
    if (!tst_fail)
        printf("INFO::[ BMPDEC PNG QUEUE OK ]\n");
    else
        printf("INFO::[ BMPDEC PNG QUEUE FAIL ]\n");


    /* Test invalid PNG */
    /* test skipped as libpng 1.6.X does not support this day/night palette change feature */
#ifdef COMMENTED_OUT
    printf("INFO::[ Testing BMPDEC PNG INVALID ]\n");
    tst_fail = tst_bmpdec_png_invalid();
    num_fail += tst_fail;
    if (!tst_fail)
        printf("INFO::[ BMPDEC PNG INVALID OK ] libpng error are ok \n");
    else
        printf("INFO::[ BMPDEC PNG INVALID FAIL ]\n");

#else
    printf("INFO::[ BMPDEC PNG INVALID skipped]\n");
#endif

    /* Test asynchronous decoding */
    printf("INFO::[ Testing BMPDEC ASYNC ]\n");
    tst_fail = tst_bmpdec_async();
    num_fail += tst_fail;
    if (!tst_fail)
        printf("INFO::[ BMPDEC ASYNC OK ]\n");
    else
        printf("INFO::[ BMPDEC ASYNC FAIL ]\n");
    /* Test memory destination */
    printf("INFO::[ Testing BMPDEC MEMORY DEST ]\n");
    tst_fail = tst_bmpdec_mem_dest();
    num_fail += tst_fail;
    if (!tst_fail)
        printf("INFO::[ BMPDEC MEMORY DEST OK ]\n");
    else
        printf("INFO::[ BMPDEC MEMORY DEST FAIL ]\n");

    svgDestroyContextBmpDec(gp_bmp_ctx);
	printf("destroyBMPDec successfull\n");
    svgCloseBmpDec();
	printf("closeBMPDec successfull\n");
    eglClose();
	printf("closeEGL successfull\n");
    return 0;
}
